# Copyright (c) 2010, 2014, Oracle and/or its affiliates. All rights reserved.
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA 

# This file includes Windows specific hacks, mostly around compiler flags

INCLUDE (CheckCSourceCompiles)
INCLUDE (CheckCXXSourceCompiles)
INCLUDE (CheckStructHasMember)
INCLUDE (CheckLibraryExists)
INCLUDE (CheckFunctionExists)
INCLUDE (CheckCCompilerFlag)
INCLUDE (CheckCSourceRuns)
INCLUDE (CheckSymbolExists)
INCLUDE (CheckTypeSize)

# Optionally read user configuration, generated by configure.js.
# This is left for backward compatibility reasons only.
INCLUDE(${CMAKE_BINARY_DIR}/win/configure.data OPTIONAL)

# avoid running system checks by using pre-cached check results
# system checks are expensive on VS since every tiny program is to be compiled in 
# a VC solution.
GET_FILENAME_COMPONENT(_SCRIPT_DIR ${CMAKE_CURRENT_LIST_FILE} PATH)
INCLUDE(${_SCRIPT_DIR}/WindowsCache.cmake)

# We require at least Visual Studio 2010 (aka 10.0) which has version nr 1600.
IF(MSVC_VERSION LESS 1600)
  MESSAGE(FATAL_ERROR "Visual Studio 2010 or newer is required!")
ENDIF()

# OS display name (version_compile_os etc).
# Used by the test suite to ignore bugs on some platforms, 
IF(CMAKE_SIZEOF_VOID_P MATCHES 8)
  SET(SYSTEM_TYPE "Win64")
  SET(MYSQL_MACHINE_TYPE "x86_64")
ELSE()
  SET(SYSTEM_TYPE "Win32")
ENDIF()

ADD_DEFINITIONS("-D_CRT_SECURE_NO_DEPRECATE")

# Target Windows Vista or later, i.e _WIN32_WINNT_VISTA
ADD_DEFINITIONS("-D_WIN32_WINNT=0x0600")
SET(CMAKE_REQUIRED_FLAGS "${CMAKE_REQUIRED_FLAGS} -D_WIN32_WINNT=0x0600")

# Speed up build process excluding unused header files
ADD_DEFINITIONS("-DWIN32_LEAN_AND_MEAN")

# We want to use std::min/std::max, not the windows.h macros
ADD_DEFINITIONS("-DNOMINMAX")
  
# Should be available on Windows Server 2003 and above.
CHECK_CXX_SOURCE_COMPILES(
"#include <Windows.h>
int main() {
  GetCurrentProcessorNumber();
  return 0;
} " HAVE_GETCURRENTPROCESSORNUMBER)
IF(HAVE_GETCURRENTPROCESSORNUMBER)
 ADD_DEFINITIONS(-DHAVE_GETCURRENTPROCESSORNUMBER=1)
ENDIF()

IF(MSVC)
  # Enable debug info also in Release build, and create PDB to be able to analyze 
  # crashes
  FOREACH(lang C CXX)
    SET(CMAKE_${lang}_FLAGS_RELEASE "${CMAKE_${lang}_FLAGS_RELEASE} /Zi")
  ENDFOREACH()
  FOREACH(type EXE SHARED MODULE)
   SET(CMAKE_{type}_LINKER_FLAGS_RELEASE "${CMAKE_${type}_LINKER_FLAGS_RELEASE} /debug")
  ENDFOREACH()
  
  # Force static runtime libraries
  # Choose C++ exception handling:
  #   If /EH is not specified, the compiler will catch structured and
  #   C++ exceptions, but will not destroy C++ objects that will go out of
  #   scope as a result of the exception.
  #   /EHsc catches C++ exceptions only and tells the compiler to assume that
  #   extern C functions never throw a C++ exception.
  FOREACH(flag 
   CMAKE_C_FLAGS_RELEASE CMAKE_C_FLAGS_RELWITHDEBINFO 
   CMAKE_C_FLAGS_DEBUG CMAKE_C_FLAGS_DEBUG_INIT 
   CMAKE_CXX_FLAGS_RELEASE  CMAKE_CXX_FLAGS_RELWITHDEBINFO
   CMAKE_CXX_FLAGS_DEBUG  CMAKE_CXX_FLAGS_DEBUG_INIT)
   STRING(REPLACE "/MD"  "/MT" "${flag}" "${${flag}}")
   SET("${flag}" "${${flag}} /EHsc")
  ENDFOREACH()
  
  # Fix CMake's predefined huge stack size
  FOREACH(type EXE SHARED MODULE)
   STRING(REGEX REPLACE "/STACK:([^ ]+)" "" CMAKE_${type}_LINKER_FLAGS "${CMAKE_${type}_LINKER_FLAGS}")
   STRING(REGEX REPLACE "/INCREMENTAL:([^ ]+)" "" CMAKE_${type}_LINKER_FLAGS_RELWITHDEBINFO "${CMAKE_${type}_LINKER_FLAGS_RELWITHDEBINFO}")
  ENDFOREACH()
  
  # Mark 32 bit executables large address aware so they can 
  # use > 2GB address space
  IF(CMAKE_SIZEOF_VOID_P MATCHES 4)
    SET(CMAKE_EXE_LINKER_FLAGS "${CMAKE_EXE_LINKER_FLAGS} /LARGEADDRESSAWARE")
  ENDIF()
  
  # Speed up multiprocessor build
  IF (MSVC_VERSION GREATER 1400)
    SET(CMAKE_C_FLAGS "${CMAKE_C_FLAGS} /MP")
    SET(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} /MP")
  ENDIF()
  
  #TODO: update the code and remove the disabled warnings
  SET(CMAKE_C_FLAGS "${CMAKE_C_FLAGS} /wd4800 /wd4805 /wd4996")
  SET(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} /wd4800 /wd4805 /wd4996 /we4099")

  IF(CMAKE_SIZEOF_VOID_P MATCHES 8)
    # _WIN64 is defined by the compiler itself. 
    # Yet, we define it here again   to work around a bug with  Intellisense 
    # described here: http://tinyurl.com/2cb428. 
    # Syntax highlighting is important for proper debugger functionality.
    ADD_DEFINITIONS("-D_WIN64")
  ENDIF()
ENDIF()

# Always link with socket library
LINK_LIBRARIES(ws2_32)
# ..also for tests
SET(CMAKE_REQUIRED_LIBRARIES ws2_32)

# IPv6 constants appeared in Vista SDK first. We need to define them in any case if they are 
# not in headers, to handle dual mode sockets correctly.
CHECK_SYMBOL_EXISTS(IPPROTO_IPV6 "winsock2.h" HAVE_IPPROTO_IPV6)
IF(NOT HAVE_IPPROTO_IPV6)
  SET(HAVE_IPPROTO_IPV6 41)
ENDIF()
CHECK_SYMBOL_EXISTS(IPV6_V6ONLY  "winsock2.h;ws2ipdef.h" HAVE_IPV6_V6ONLY)
IF(NOT HAVE_IPV6_V6ONLY)
  SET(IPV6_V6ONLY 27)
ENDIF()

# Some standard functions exist there under different
# names (e.g popen is _popen or strok_r is _strtok_s)
# If a replacement function exists, HAVE_FUNCTION is
# defined to 1. CMake variable <function_name> will also
# be defined to the replacement name.
# So for example, CHECK_FUNCTION_REPLACEMENT(popen _popen)
# will define HAVE_POPEN to 1 and set variable named popen
# to _popen. If the header template, one needs to have
# cmakedefine popen @popen@ which will expand to 
# define popen _popen after CONFIGURE_FILE

MACRO(CHECK_SYMBOL_REPLACEMENT symbol replacement header)
  STRING(TOUPPER ${symbol} symbol_upper)
  CHECK_SYMBOL_EXISTS(${symbol} ${header} HAVE_${symbol_upper})
  IF(NOT HAVE_${symbol_upper})
    CHECK_SYMBOL_EXISTS(${replacement} ${header} HAVE_${replacement})
    IF(HAVE_${replacement})
      SET(HAVE_${symbol_upper} 1)
      SET(${symbol} ${replacement})
    ENDIF()
  ENDIF()
ENDMACRO()

CHECK_SYMBOL_REPLACEMENT(S_IROTH _S_IREAD sys/stat.h)
CHECK_SYMBOL_REPLACEMENT(S_IFIFO _S_IFIFO sys/stat.h)
CHECK_SYMBOL_REPLACEMENT(SIGQUIT SIGTERM signal.h)
CHECK_SYMBOL_REPLACEMENT(SIGPIPE SIGINT signal.h)
CHECK_SYMBOL_EXISTS(isnan math.h HAVE_ISNAN)
IF(NOT HAVE_ISNAN)
  CHECK_SYMBOL_REPLACEMENT(isnan _isnan float.h)
ENDIF()

# See http://msdn.microsoft.com/en-us/library/ms235384.aspx
# about POSIX functions and their Windows replacements

# Windows replacement functions.
SET(alloca _alloca)
SET(finite _finite)
SET(popen _popen)
SET(pclose _pclose)
SET(strcasecmp _stricmp)
SET(strncasecmp _strnicmp)
SET(strtoll _strtoi64)
SET(strtoull _strtoui64)
SET(snprintf _snprintf)
SET(vsnprintf _vsnprintf)

# Windows replacement functions where the POSIX name is deprecated.
SET(access _access)
SET(chdir _chdir)
SET(chmod _chmod)
SET(dup _dup)
SET(fdopen _fdopen)
SET(fileno _fileno)
SET(getcwd _getcwd)
SET(isatty _isatty)
SET(mkdir _mkdir)
SET(putenv _putenv)
SET(read _read)
SET(rmdir _rmdir)
SET(strdup _strdup)
SET(stricmp _stricmp)
SET(tzset _tzset)
SET(umask _umask)
SET(unlink _unlink)

# Windows security enchanced replacement functions.
SET(strtok_r strtok_s)

CHECK_TYPE_SIZE(ssize_t SIZE_OF_SSIZE_T)
IF(NOT HAVE_SIZE_OF_SSIZE_T)
 SET(ssize_t SSIZE_T)
ENDIF()

SET(FN_NO_CASE_SENSE 1)
